package io.axoniq.dataprotection.cryptoengine.vault;

import java.io.IOException;
import java.lang.invoke.MethodHandles;
import java.util.UUID;

import org.junit.Assert;
import org.junit.Assume;
import org.junit.Before;
import org.junit.Test;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import io.axoniq.dataprotection.cryptoengine.AbstractEngineTestSet;
import io.axoniq.dataprotection.cryptoengine.CryptoEngine;

/**
 * Tests of the VaultCryptoEngine by using VAULT_ENGINE_VERSION 2 against Vault Server.
 * The VAULT_ADDRESS and ROOT_TOKEN are hard coded in the test Utils class.
 * The simplest way to run this test:
 * - run a dev-mode Vault server on localhost (vault server -dev)
 * - copy the root token from the logging and paste in in the ROOT_TOKEN constant
 *
 * The tests can also be run against a full-fledged Vault server. We're using an OkHttpClient that
 * allows untrusted certs, so it's also possible to run the tests again a Vault server that uses
 * self-signed certificates.
 *
 * Please note that this test code uses the BetterCloud Vault client to do authentication and policy
 * management. This client is exclusively used in the test code, not in the module's production code,
 * because of performance constraints in this client.
 */
public class VaultApiVersion2Test extends AbstractEngineTestSet {
    private static final Logger logger = LoggerFactory.getLogger(MethodHandles.lookup().lookupClass());
    private static final int VAULT_ENGINE_VERSION = 2;

    private static VaultCryptoEngine vaultCryptoEngine;

    static {
        try {
                vaultCryptoEngine = Utils.initVaultCryptoEngine(VAULT_ENGINE_VERSION);
        } catch (Exception ex) {
            logger.error("Unable to configure Vault, Vault API 2 tests will be disabled.", ex);
            vaultCryptoEngine = null;
        }
    }

    @Before
    public void beforeMethod() {
        Assume.assumeTrue(vaultCryptoEngine != null);
    }

    protected CryptoEngine getCryptoEngine() {
        return vaultCryptoEngine;
    }

    /**
     * In addition to the standard tests in AbstractEngineTestSet, we want to verify that
     * we don't overwrite keys; this depends on having the right policy configured.
     */
    @Test
    public void updatesDontHappen() throws IOException {
        UUID id = UUID.randomUUID();
        vaultCryptoEngine.putKey(id.toString(), Utils.generateKey());
        try {
            vaultCryptoEngine.putKey(id.toString(), Utils.generateKey());
            Assert.fail("Key got updated; should have received 403 exception instead.");
        } catch (PermissionDeniedException ex) {
        }
    }
}
