/*
 * Copyright (c) 2017-2018. AxonIQ B.V.
 */
package io.axoniq.dataprotection.fieldencryption;

import io.axoniq.dataprotection.api.DataSubjectId;
import io.axoniq.dataprotection.api.FieldEncrypter;
import io.axoniq.dataprotection.api.PersonalData;
import io.axoniq.dataprotection.cryptoengine.CryptoEngine;
import io.axoniq.dataprotection.cryptoengine.InMemoryCryptoEngine;
import lombok.Data;
import org.axonframework.serialization.Serializer;
import org.axonframework.serialization.xml.XStreamSerializer;
import org.fluttercode.datafactory.impl.DataFactory;
import org.jetbrains.annotations.NotNull;
import org.junit.Assert;
import org.junit.Before;
import org.junit.Test;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.Iterator;
import java.util.List;
import java.util.ListIterator;
import java.util.Set;
import java.util.SortedSet;
import java.util.Spliterator;
import java.util.TreeSet;
import java.util.concurrent.ThreadLocalRandom;
import java.util.function.Consumer;
import java.util.function.Predicate;
import java.util.stream.Stream;

/**
 * Contains tests of encryption in objects in special collections (immutables etc.)
 */
public class SpecialCollectionsTest {

    private CryptoEngine cryptoEngine;
    private FieldEncrypter fieldEncrypter;
    private Serializer serializer = XStreamSerializer.defaultSerializer();
    private DataFactory dataFactory;

    @Before
    public void setUp() throws Exception {
        cryptoEngine = new InMemoryCryptoEngine();
        fieldEncrypter = new FieldEncrypter(cryptoEngine);
        dataFactory = new DataFactory();
        dataFactory.randomize(ThreadLocalRandom.current().nextInt());
    }

    @Test
    public void nullMustStayNull() {
        A a = new A(1L, null);
        fieldEncrypter.encrypt(a);
        Assert.assertNull(a.data);
        fieldEncrypter.decrypt(a);
        Assert.assertNull(a.data);
    }

    @Test
    public void emptyList() {
        A a = new A(1L, Collections.EMPTY_LIST);
        fieldEncrypter.encrypt(a);
        Assert.assertTrue(a.data.isEmpty());
        fieldEncrypter.decrypt(a);
        Assert.assertTrue(a.data.isEmpty());
    }

    @Test
    public void singletonList() {
        A a = new A(1L, Collections.singletonList("TEST"));
        fieldEncrypter.encrypt(a);
        Assert.assertFalse(a.data.get(0).equals("TEST"));
        fieldEncrypter.decrypt(a);
        Assert.assertTrue(a.data.get(0).equals("TEST"));
    }

    @Test
    public void singletonSet() {
        B a = new B(1L, Collections.singleton("TEST"));
        fieldEncrypter.encrypt(a);
        Assert.assertFalse(a.data.contains("TEST"));
        fieldEncrypter.decrypt(a);
        Assert.assertTrue(a.data.contains("TEST"));
    }

    @Test
    public void unmodifiableList() {
        List<String> list = new ArrayList<>();
        list.add("TEST1");
        list.add("TEST2");
        A a = new A(1L, Collections.unmodifiableList(list));
        String serialized1 = serializer.serialize(a, String.class).getData();
        System.out.println(serialized1);
        Assert.assertTrue(serialized1.contains("TEST"));
        fieldEncrypter.encrypt(a);
        String serialized2 = serializer.serialize(a, String.class).getData();
        System.out.println(serialized2);
        Assert.assertFalse(serialized2.contains("TEST"));
        fieldEncrypter.decrypt(a);
        String serialized3 = serializer.serialize(a, String.class).getData();
        Assert.assertEquals(serialized1, serialized3);
    }

    @Test
    public void arrayAsList() {
        A a = new A(1L, Arrays.asList("TEST1", "TEST2"));
        String serialized1 = serializer.serialize(a, String.class).getData();
        System.out.println(serialized1);
        Assert.assertTrue(serialized1.contains("TEST"));
        fieldEncrypter.encrypt(a);
        String serialized2 = serializer.serialize(a, String.class).getData();
        System.out.println(serialized2);
        Assert.assertFalse(serialized2.contains("TEST"));
        fieldEncrypter.decrypt(a);
        String serialized3 = serializer.serialize(a, String.class).getData();
        Assert.assertEquals(serialized1, serialized3);
    }

    @Test
    public void synchronizedUnmodifiableList() {
        List<String> list = new ArrayList<>();
        list.add("TEST1");
        list.add("TEST2");
        A a = new A(1L, Collections.synchronizedList(Collections.unmodifiableList(list)));
        String serialized1 = serializer.serialize(a, String.class).getData();
        System.out.println(serialized1);
        Assert.assertTrue(serialized1.contains("TEST"));
        fieldEncrypter.encrypt(a);
        String serialized2 = serializer.serialize(a, String.class).getData();
        System.out.println(serialized2);
        Assert.assertFalse(serialized2.contains("TEST"));
        fieldEncrypter.decrypt(a);
        String serialized3 = serializer.serialize(a, String.class).getData();
        Assert.assertEquals(serialized1, serialized3);
    }

    @Test
    public void checkedUnmodifiableSortedSet() {
        SortedSet<String> set = new TreeSet<>();
        set.add("TEST1");
        set.add("TEST2");
        B a = new B(1L, Collections.checkedSortedSet(Collections.unmodifiableSortedSet(set), String.class));
        String serialized1 = serializer.serialize(a, String.class).getData();
        System.out.println(serialized1);
        Assert.assertTrue(serialized1.contains("TEST"));
        fieldEncrypter.encrypt(a);
        String serialized2 = serializer.serialize(a, String.class).getData();
        System.out.println(serialized2);
        Assert.assertFalse(serialized2.contains("TEST"));
        fieldEncrypter.decrypt(a);
        String serialized3 = serializer.serialize(a, String.class).getData();
        Assert.assertEquals(serialized1, serialized3);
    }

    @Test
    public void customImplementationList() {
        List<String> list = new CustomImmutableList("TEST1", "TEST2");
        A a = new A(1L, list);
        String serialized1 = serializer.serialize(a, String.class).getData();
        System.out.println(serialized1);
        Assert.assertTrue(serialized1.contains("TEST"));
        fieldEncrypter.encrypt(a);
        String serialized2 = serializer.serialize(a, String.class).getData();
        System.out.println(serialized2);
        Assert.assertFalse(serialized2.contains("TEST"));
        fieldEncrypter.decrypt(a);
        String serialized3 = serializer.serialize(a, String.class).getData();
        Assert.assertEquals(a, new A(1L, list));
        // we have modified the type of List, so the serialized forms are not be equal anymore
        Assert.assertNotEquals(serialized1, serialized3);
    }

    @Data
    public static class A {
        @DataSubjectId
        private final long id;
        @PersonalData
        private final List<String> data;
    }

    @Data
    public static class B {
        @DataSubjectId
        private final long id;
        @PersonalData
        private final Set<String> data;
    }

    private static class CustomImmutableList implements List<String> {

        private final List<String> backend;

        public CustomImmutableList(String... values) {
            this.backend = Arrays.asList(values);
        }

        @Override
        public int size() {
            return backend.size();
        }

        @Override
        public boolean isEmpty() {
            return backend.isEmpty();
        }

        @Override
        public boolean contains(Object o) {
            return backend.contains(o);
        }

        @NotNull
        @Override
        public Iterator<String> iterator() {
            return backend.iterator();
        }

        @NotNull
        @Override
        public Object[] toArray() {
            return backend.toArray();
        }

        @NotNull
        @Override
        public <T> T[] toArray(@NotNull T[] a) {
            return backend.toArray(a);
        }

        @Override
        public boolean add(String s) {
            throw new UnsupportedOperationException("Immutable list");
        }

        @Override
        public boolean remove(Object o) {
            throw new UnsupportedOperationException("Immutable list");
        }

        @Override
        public boolean containsAll(@NotNull Collection<?> c) {
            return backend.containsAll(c);
        }

        @Override
        public boolean addAll(@NotNull Collection<? extends String> c) {
            throw new UnsupportedOperationException("Immutable list");
        }

        @Override
        public boolean addAll(int index, @NotNull Collection<? extends String> c) {
            throw new UnsupportedOperationException("Immutable list");
        }

        @Override
        public boolean removeAll(@NotNull Collection<?> c) {
            throw new UnsupportedOperationException("Immutable list");
        }

        @Override
        public boolean retainAll(@NotNull Collection<?> c) {
            throw new UnsupportedOperationException("Immutable list");
        }

        @Override
        public void clear() {
            throw new UnsupportedOperationException("Immutable list");
        }

        @Override
        public boolean equals(Object o) {
            return backend.equals(o);
        }

        @Override
        public int hashCode() {
            return backend.hashCode();
        }

        @Override
        public String get(int index) {
            return backend.get(index);
        }

        @Override
        public String set(int index, String element) {
            throw new UnsupportedOperationException("Immutable list");
        }

        @Override
        public void add(int index, String element) {
            throw new UnsupportedOperationException("Immutable list");
        }

        @Override
        public String remove(int index) {
            throw new UnsupportedOperationException("Immutable list");
        }

        @Override
        public int indexOf(Object o) {
            return backend.indexOf(o);
        }

        @Override
        public int lastIndexOf(Object o) {
            return backend.lastIndexOf(o);
        }

        @NotNull
        @Override
        public ListIterator<String> listIterator() {
            return backend.listIterator();
        }

        @NotNull
        @Override
        public ListIterator<String> listIterator(int index) {
            return backend.listIterator(index);
        }

        @NotNull
        @Override
        public List<String> subList(int fromIndex, int toIndex) {
            return backend.subList(fromIndex, toIndex);
        }

        @Override
        public Spliterator<String> spliterator() {
            return backend.spliterator();
        }

        @Override
        public boolean removeIf(Predicate<? super String> filter) {
            throw new UnsupportedOperationException("Immutable list");
        }

        @Override
        public Stream<String> stream() {
            return backend.stream();
        }

        @Override
        public Stream<String> parallelStream() {
            return backend.parallelStream();
        }

        @Override
        public void forEach(Consumer<? super String> action) {
            backend.forEach(action);
        }
    }
}
